import express from 'express';
import http from 'http';
import { Server as SocketIOServer } from 'socket.io';
import cors from 'cors';
import path from 'path';
import sequelize from './db/connection.js';
import routesUser from './routes/user.js';
import routesCarrito from './routes/carrito.js';
import routesProductos from './routes/productos.js';
import routesMovimientos from './routes/movimientos.js';
import routesUploadArchive from './routes/upload_archive.js';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

class Server {
    private app: express.Application;
    private port: string;
    private server: http.Server;
    public io: SocketIOServer;

    constructor() {
        this.app = express();
        this.port = process.env.PORT || '3001';

        // Crear servidor HTTP y Socket.IO
        this.server = http.createServer(this.app);
        this.io = new SocketIOServer(this.server, {
            cors: {
                origin: [
                    'http://localhost:4200',
                    'https://tesoreria.distribuyendo.cl',
                    'https://ukhaisushi.distribuyendo.cl',
                    'https://ukhaisushi.cl'
                ],
                methods: ['GET', 'POST', 'PUT', 'DELETE'],
                credentials: true
            }
        });

        // Hacer disponible el io para los controladores
        this.app.set('io', this.io);

        this.middlewares();
        this.routes();
        this.socketEvents(); // configuración sockets
        this.dbConnection();
    }

    async dbConnection() {
        try {
            await sequelize.authenticate();
            await sequelize.sync();
            console.clear();
            console.log('\x1b[32m%s\x1b[0m', '✅ Servidor Corriendo');
            console.log('\x1b[34m%s\x1b[0m', '✅ Conectado a la Base de Datos');
            this.listen();
        } catch (error) {
            console.error('\x1b[31m%s\x1b[0m', '❌ Error al conectar la BD');
            console.error(error);
        }
    }

    listen() {
        this.server.listen(this.port, () => {
            console.log(`🚀 Servidor Corriendo en https://apiukhai.distribuyendo.cl`);
        });
    }

    routes() {
        this.app.get('/', (req, res) => {
            res.json({
                message: "🎉 ¡Bienvenido al backend de UkhaiSushi! 🎉",
                rutas: {
                    usuarios: "/users",
                    subirArchivo: "/upload",
                }
            });
        });

        this.app.use('/users', routesUser);
        this.app.use('/carrito', routesCarrito);
        this.app.use('/productos', routesProductos);
        this.app.use('/movimientos', routesMovimientos);
        this.app.use('/upload', routesUploadArchive);
        this.app.use("/img", express.static(path.join(__dirname, "img")));

        this.app.use((req, res, next) => {
            res.setTimeout(10000, () => {
                console.log("⏳ Timeout alcanzado en", req.url);
                res.status(503).send("Servidor ocupado. Intenta de nuevo.");
            });
            next();
        });

        this.app.use((req, res) => {
            res.status(404).json({
                error: "⛔ Ruta no encontrada",
                message: `La ruta '${req.originalUrl}' no existe en el servidor.`
            });
        });
    }

    middlewares() {
        const allowedOrigins = ['http://localhost:4200', 'https://tesoreria.distribuyendo.cl', 'https://ukhaisushi.distribuyendo.cl', 'https://ukhaisushi.cl'];

        this.app.use(cors({
            origin: function (origin, callback) {
                if (!origin || allowedOrigins.includes(origin)) {
                    callback(null, true);
                } else {
                    callback(new Error('CORS not allowed'));
                }
            },
            methods: ['GET', 'POST', 'PUT', 'DELETE'],
            allowedHeaders: ['Content-Type', 'Authorization'],
            credentials: true,
        }));

        this.app.use(express.json());
    }

    socketEvents() {
        this.io.on('connection', (socket) => {
            console.log('🔌 Cliente conectado:', socket.id);

            socket.on('disconnect', () => {
                console.log('❌ Cliente desconectado:', socket.id);
            });

            // Puedes definir aquí otros eventos personalizados si lo deseas
        });
    }
}

export default Server;